<?php
/**
 * Plugin Name: Airgentic AI Chat Overlay
 * Plugin URI: https://www.airgentic.com
 * Description: Adds the Airgentic chat and search overlay to your WordPress site.
 * Version: 1.0.0
 * Author: Airgentic Pty Ltd
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: airgentic
 * Requires at least: 5.8
 * Tested up to: 6.8
 * Requires PHP: 7.4
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

// Define plugin constants
define( 'AIRGENTIC_VERSION', '1.0.0' );
define( 'AIRGENTIC_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'AIRGENTIC_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * Main Airgentic plugin class
 */
final class Airgentic {

	/**
	 * Script handle for wp_enqueue_script
	 *
	 * @var string
	 */
	private const SCRIPT_HANDLE = 'airgentic-widget';

	/**
	 * Plugin instance
	 *
	 * @var Airgentic
	 */
	private static $instance = null;

	/**
	 * Plugin options
	 *
	 * @var array
	 */
	private $options = array();

	/**
	 * Get plugin instance
	 *
	 * @return Airgentic
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->load_dependencies();
		$this->init_hooks();
	}

	/**
	 * Load required dependencies
	 */
	private function load_dependencies() {
		if ( is_admin() ) {
			require_once AIRGENTIC_PLUGIN_DIR . 'includes/admin-settings.php';
		}
	}

	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks() {
		// Front-end hooks
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_script' ) );
		add_filter( 'script_loader_tag', array( $this, 'add_data_attributes_to_script' ), 10, 3 );
		add_action( 'wp_footer', array( $this, 'output_widget_container' ) );

		// Shortcode
		add_shortcode( 'airgentic_service_hub', array( $this, 'service_hub_shortcode' ) );
	}

	/**
	 * Get plugin options
	 *
	 * @return array
	 */
	private function get_options() {
		if ( empty( $this->options ) ) {
			$this->options = get_option( 'airgentic_options', array() );
		}
		return $this->options;
	}

	/**
	 * Enqueue the Airgentic script
	 */
	public function enqueue_script() {
		$options = $this->get_options();

		// Check if plugin is enabled and configured
		if ( empty( $options['enable_on_front'] ) ) {
			return;
		}

		if ( empty( $options['account_id'] ) || empty( $options['service_id'] ) ) {
			return;
		}

		// Enqueue the Airgentic script
		wp_enqueue_script(
			self::SCRIPT_HANDLE,
			'https://chat.airgentic.com/_js/airgentic-1.4.js',
			array(),
			AIRGENTIC_VERSION,
			array( 'in_footer' => false )
		);
	}

	/**
	 * Add data attributes to the Airgentic script tag
	 *
	 * @param string $tag    The script tag.
	 * @param string $handle The script handle.
	 * @param string $src    The script source URL.
	 * @return string Modified script tag with data attributes.
	 */
	public function add_data_attributes_to_script( $tag, $handle, $src ) {
		// Only modify our script
		if ( self::SCRIPT_HANDLE !== $handle ) {
			return $tag;
		}

		$options = $this->get_options();

		// Build data attributes
		$data_attrs = 'data-account-id="' . esc_attr( $options['account_id'] ) . '" ' .
		              'data-service-id="' . esc_attr( $options['service_id'] ) . '"';

		// Add optional search element IDs
		if ( ! empty( $options['search_input_id'] ) ) {
			$data_attrs .= ' data-search-input-id="' . esc_attr( $options['search_input_id'] ) . '"';
		}
		if ( ! empty( $options['search_button_id'] ) ) {
			$data_attrs .= ' data-search-button-id="' . esc_attr( $options['search_button_id'] ) . '"';
		}

		// Modify the WordPress-generated tag:
		// 1. Change ID from 'airgentic-widget-js' to 'airgentic-script'
		$tag = str_replace( 'id="airgentic-widget-js"', 'id="airgentic-script"', $tag );
		$tag = str_replace( "id='airgentic-widget-js'", "id='airgentic-script'", $tag );
		
		// 2. Add async attribute after opening script tag
		$tag = str_replace( '<script ', '<script async ', $tag );
		
		// 3. Remove version query parameter
		$tag = preg_replace( '/\?ver=[^\'"\s>]+/', '', $tag );
		
		// 4. Add data attributes and defer before src attribute
		$tag = str_replace( ' src=', ' ' . $data_attrs . ' defer src=', $tag );

		return $tag;
	}

	/**
	 * Output the widget container div in the footer
	 */
	public function output_widget_container() {
		$options = $this->get_options();

		// Check if plugin is enabled
		if ( empty( $options['enable_on_front'] ) ) {
			return;
		}

		if ( empty( $options['account_id'] ) || empty( $options['service_id'] ) ) {
			return;
		}

		echo '<div id="airgentic"></div>' . "\n";
	}

	/**
	 * Service Hub shortcode handler
	 *
	 * @return string
	 */
	public function service_hub_shortcode() {
		return '<div id="airgentic"></div>';
	}
}

/**
 * Initialize the plugin
 */
function airgentic_init() {
	return Airgentic::instance();
}

// Start the plugin
add_action( 'plugins_loaded', 'airgentic_init' );

